/*
	Author: wamesbury
	Created: 2020-04-17
	Purpose: Claim Mapping editor & sub-editors
	History:
	2024-11-05 DJA THOR-463 Move "Use Certifying Physician" to Referring Provider section & Relabel as "Include Primary on POC"
	2024-07-11 RCB THOR-330 Added CombineVisitByDateRevenueCode
	2023-05-25 RCB HS-13445 Added IncludeRefPhy and AllowDupPhy properties
	2023-01-06 WRA ND-12440 Added CreateCopyOfClaimForInsurance property
	2022-10-04 DVDH ND-12127 Added HasDCNForTransferredCharges, HasDCNForReplacementClaims
	2022-08-01 JCF ND-11888 Added IncludeMedicaidProviderNumber
	2021-12-15 WRA ND-11335 Added TypeOfBillOverride
	2021-12-13 RCB ND-11669 Added includeRelationship
	2021-11-03 RCB ND-11618 Added DoNotCombineLOC
	2021-10-28 RCB ND-11438 Added CombineSupplyCharges
	2021-10-04 RCB ND-11542 Added BackgroundImagePathAlt
	2021-09-10 RCB ND-11422 Added requireOrderingProvider and RequireCOB fields
	2021-09-02 RCB ND-11419 Added RefProviderOtherId field
	2021-08-27 WRA ND-11419 Added OtherProviderIdentifier field
	2021-08-10 WRA ND-11404 Added Statement Dates fields to Patient Statement. Changed statement payment due label from "x days from Aging Date" to "x days from Process Date"
	2021-05-27 RMW ND-11123 tightened Default Hipps Code regex to match the HCPCS code field for charges
	2021-05-17 RMW ND-11228 Added ProviderSignature field
	2021-05-13 RMW ND-11169 Fixed top service line issue
	2021-04-15 WRA ND-10487 Added CombineMedicationCharges
	2021-04-01 WRA ND-11033 Enabled additional coding for paper claims
	2021-01-19 WRA ND-10634 Added PatientStatementLogoPath
	2020-12-08 RMW ND-10622 Added Scenario 9 and Default HIPPS Code fields
	2020-06-10 WRA ND-10216 Added Type of Care field.
*/

define([
	"dojo/_base/declare",
	"thornberry/object/ObjectEditor",
	"thornberry/object/ObjectModel",
	"thornberry/serverRequest",
	"thornberry/object/TableManager",
	"thornberry/object/template",
	"dojo/_base/array",
	"dojo/_base/lang",
	"dojo/aspect",
	"dojo/store/Memory",
	"dojo/dom-construct",
	"dojo/dom-style",
	"dojo/query",
	"dojo/Deferred",
	"dijit/form/NumberTextBox",
	"dijit/form/Button",
	"dijit/Dialog",
	"dijit/form/MultiSelect",
	"gridx/Grid",
	"gridx/core/model/cache/Sync",
	"gridx/modules/select/Row",
	"gridx/modules/VirtualVScroller",
	"gridx/modules/IndirectSelect",
	"gridx/modules/RowHeader",
	"gridx/modules/SingleSort",
	"dijit/_WidgetBase",
	"dijit/_TemplatedMixin",
	"dijit/_WidgetsInTemplateMixin"
], function(declare, ObjectEditor, ObjectModel, serverRequest, TableManager, template, arrayUtils, lang, aspect, Memory,
			domConstruct, domStyle, query, Deferred, NumberTextBox, Button,	Dialog, MultiSelect,
			Grid, Cache, SelectRow, VScroller, IndirectSelect, RowHeader, SingleSort,
			_WidgetBase, _TemplatedMixin, _WidgetsInTemplateMixin) {

	var revenueCodeRegEx = "\\d{4}";
	var revenueCodeInvalidMessage = "Revenue codes are 4 digits";
	var insuranceTypeOptions = {};
	TableManager.get("FillInGroup~20605").query().forEach(function(type) {
		insuranceTypeOptions[type.id] = type.name;
	});
	if (hospiceAgency != 1) {
		delete insuranceTypeOptions[2];
	}

	var MINDATE = new Date("12/31/1840");

	var claimMappingModel = new ObjectModel({ propertyDefinitions: [
		{ name: "id", type: "text" },
		{ name: "Name", type: "text", maxLength: 50, required: true },
		{ name: "Status", type: "enum", displayIndex: { "active": "Active", "inactive": "Inactive" }, required: true },
		{ name: "InsuranceType", type: "enum", displayIndex: insuranceTypeOptions, required: true },
		{ name: "ClaimForm", type: "tableItem", tableName: "Tables.Insurance.ClaimForm", required: true },
		{ name: "PrintWithFormImage", type: "boolean" },
		{ name: "BackgroundImagePathAlt", type: "boolean" },
		{ name: "CreateCopyOfClaimForInsurance", type: "tableItem", tableName: "Tables.Insurance" },
		
		// Generic electronic options
		{ name: "InterchangeSenderID", type: "text", required: true, maxLength: 15 },
		{ name: "InterchangeReceiverQualifier", type: "text", required: true, maxLength: 2 },
		{ name: "InterchangeReceiverID", type: "text", required: true, maxLength: 15 },
		{ name: "RequestInterchangeAcknowledgement", type: "boolean" },
		{ name: "ApplicationSenderID", type: "text", maxLength: 15 },
		{ name: "ApplicationReceiverID", type: "text", maxLength: 15 },

		// Patient Statement
		{ name: "PayToName", type: "text", maxLength: 80 },
		{ name: "PayToAddressStreet1", type: "text" },
		{ name: "PayToAddressStreet2", type: "text" },
		{ name: "PayToAddressCity", type: "text" },
		{ name: "PayToAddressState", type: "text" },
		{ name: "PayToAddressZip", type: "text", maxLength: 10 },
		{ name: "PaymentPhoneNumber", type: "text", maxLength: 25 },
		{ name: "PaymentWebsite", type: "text", maxLength: 80 },
		{ name: "PaymentDueDate", type: "text" },
		{ name: "PatientStatementProviderDetailLocation", type: "tableItem", required: true, tableName: "FillInGroup~20613" },
		{ name: "PatientStatementLogoPath", type: "text", maxLength: 100 },

		// 270 options
		{ name: "CustomInformationSourceName", type: "text", maxLength: 60 },
		{ name: "InformationSourceIdQualifier", type: "text", maxLength: 2 },
		{ name: "CustomInformationSourceId", type: "text", maxLength: 80 },
		{ name: "InformationReceiverIdQualifier", type: "text", maxLength: 2 },
		{ name: "CustomInformationReceiverID", type: "text", maxLength: 80 },
		{ name: "InformationReceiverIndividualOrOrganizationName", type: "text", maxLength: 60 },
		{ name: "InformationReceiverIdentificationCodeQualifier", type: "text", maxLength: 2 },
		{ name: "InformationReceiverIdentificationCode", type: "text", maxLength: 80 },
		{ name: "IncludeTaxonomyCode", type: "boolean" },
		{ name: "CustomTaxonomyCode", type: "text", maxLength: 50 },
		{ name: "ServiceTypeCodes", type: "text" },

		// Claim options
		{ name: "ProviderDetailLocation", type: "tableItem", required: true, tableName: "FillInGroup~20613" },
		{ name: "TaxonomyCodeQualifier", type: "text", maxLength: 2 },
		{ name: "TaxonomyCode", type: "text", maxLength: 10 },
		{ name: "TypeOfCare", type: "tableItem", required: true, tableName: "FillInGroup~20636" },
		{ name: "TypeOfBillOverride", type: "text", maxLength: 3 },
		{ name: "ProviderSignature", type: "tableItem", required: false, tableName: "FillInGroup~20646" },
		{ name: "ProviderSigOnFileText", type: "text", maxLength: 20 },
		{ name: "ProviderSigOnFileDate", type: "tableItem", tableName: "FillInGroup~20651" },
		{ name: "StatementFromDate", type: "tableItem", required: true, tableName: "FillInGroup~20614" },
		{ name: "StatementThroughDate", type: "tableItem", required: true, tableName: "FillInGroup~20615" },
		{ name: "AllowMultiplePriorAuth", type: "boolean" },
		{ name: "IncludeMedicaidProviderNumber", type: "boolean" },
		{ name: "TreatmentAuthorizationCode", type: "tableItem", required: true, tableName: "FillInGroup~20616" },
		{ name: "DiagnosesSource", type: "tableItem", required: true, tableName: "FillInGroup~20617" },
		{ name: "EciCodes", type: "tableItem", required: true, tableName: "FillInGroup~20645" },
		{ name: "ProceduresSource", type: "tableItem", required: true, tableName: "FillInGroup~20617" },
		{ name: "TopServiceLineWithHipps", type: "boolean" },
		{ name: "TopServiceLineRevenueCode", type: "text", maxLength: 4, required: true },
		{ name: "TopServiceLineQuantity", type: "text", required: true },
		{ name: "TopServiceLineChargeAmount", type: "text", required: true },
		{ name: "DefaultHippsCode", type: "text", maxLength: 5 },
		{ name: "WeeklyReportingOfGIP", type: "boolean" },
		{ name: "RevenueCodeIncludeLeadingZero", type: "boolean" },
		{ name: "CombineMedicationCharges", type: "boolean" },
		{ name: "CombineSupplyCharges", type: "boolean" },
		{ name: "DoNotCombineLOC", type: "boolean" },
		{ name: "requireOrderingProvider", type: "boolean" },
		{ name: "RequireCOB", type: "boolean" },
		{ name: "includeRelationship", type: "boolean" },
		{ name: "ServiceUnitCount", type: "tableItem", tableName: "FillInGroup~20618" },
		{ name: "ServiceUnitsPerCharge", type: "text" },
		{ name: "ChargeAmount", type: "tableItem", tableName: "FillInGroup~20619" },
		{ name: "CopayAsAmountPaid", type: "boolean" },
		{ name: "CurrentIllnessDate", type: "tableItem", tableName: "FillInGroup~20649" },
		{ name: "DefaultNpiSource", type: "tableItem", tableName: "FillInGroup~20652" },
		{ name: "IncludeCertifyingPhysician", type: "boolean" },
		{ name: "IncludeAttendingProviderTaxonomyCode", type: "boolean" },
		{ name: "AttendingProviderOtherId", type: "tableItem", tableName: "FillInGroup~20648" },
		{ name: "IncludeRefPhy", type: "boolean" },
		{ name: "RefProviderOtherId", type: "tableItem", tableName: "FillInGroup~20648" },
		{ name: "AllowDupPhy", type: "boolean" },
		{ name: "IncludeRoomAndBoardFacility", type: "boolean" },
		{ name: "CombineVisitByDateRevenueCode", type: "boolean" },
		{ name: "CombineRoomAndBoard", type: "boolean" },
		{ name: "ExcludeChargeTypes", type: "list", itemType: "tableItem", tableName: "CustomChargeTypeList" },
		{ name: "DoNotOverwriteHCPCS", type: "boolean" },

		// Paper claim options
		{ name: "OtherProviderIdentifier", type: "tableItem", tableName: "FillInGroup~20647" },
		{ name: "BackgroundImagePathAlt", type: "boolean" },

		// Scenarios
		{ name: "Scenario1", type: "boolean" }, // Include Occurrence Code 61 (or 62) w/ Inpatient Discharge Date when inpatient discharge has occurred within 14 days of bill period start
		{ name: "Scenario2", type: "boolean" }, // Include Occurrence Span Code 77 w/ Uncovered Date Range when NOE was submitted late
		{ name: "Scenario3", type: "boolean" }, // Include Occurrence Span Code M2 w/ Respite Care Date Range for each set Respite Care days when multiple sets of respite days are included on the claim
		{ name: "Scenario4", type: "boolean" }, // Include Occurrence Code 42 w/ Discharge Date when Discharge Reason has an HIS Reason for Discharge that is "02 - Revoked"
		{ name: "Scenario5", type: "boolean" }, // Include Condition Code 52 when Discharge Reason has an HIS Reason for Discharge that is "04 - Moved out of hospice service area"
		{ name: "Scenario6", type: "boolean" }, // Include Condition Code H2 when Discharge Reason has an HIS Reason for Discharge that is "06 - Discharge for cause"
		{ name: "Scenario7", type: "boolean" }, // Include Remark w/ Discharge Reason & Date when Discharge Reason has an HIS Reason for Discharge that is NOT "01 - Expired"
		{ name: "Scenario8", type: "boolean" }, // Include CBSA in value code(s) 61 and/or G8 based on Level of Care provided to patient
		{ name: "Scenario10", type: "boolean" }, // Include Condition Code 47 when B-Transfer from another HHA selected as Source of Admission
		
		// DCN
		{ name: "HasDCNForTransferredCharges", type: "boolean" },
		{ name: "HasDCNForReplacementClaims", type: "boolean" }
	]});

	var claimMappingTemplate = {
		noForm: false,
		containers: [{
			sections: [{
				hideDivider: false,
				rows: [{
					fields: [{
						name: "Insurances",
						displayName: "Insurances:",
						labelSize: "medium-large"
					}, {
						name: "copyButton",
						noField: true
					}]
				}]
			}, {
				hideDivider: false,
				rows: [{
					fields: [{
						name: "Name",
						displayName: "Name:",
						fieldSize: "largest",
						labelSize: "medium-large"
					}, {
						name: "Status",
						displayName: "Status:",
						labelSize: "smaller"
					}]
				}, {
					fields: [{
						name: "InsuranceType",
						displayName: "Insurance Type:",
						labelSize: "medium-large"
					}]
				}, {
					fields: [{
						name: "ClaimForm",
						displayName: "Claim Form:",
						fieldSize: "larger",
						labelSize: "medium-large"
					}, {
						name: "PrintWithFormImage",
						displayName: "Print w/ Form Image?",
						labelSize: "large",
						containerStyle: "display: none;"
					}, {
						name: "BackgroundImagePathAlt",
						displayName: "Use Alt Background?",
						labelSize: "large",
						containerStyle: "display: none;"
					}, {
						name: "CreateCopyOfClaimForInsurance",
						displayName: "Create copy of claim for insurance, if present:",
						labelSize: "grand",
						containerStyle: "display: none;"
					}]
				}]
			}]
		}, {
			attachPoint: "genericElectronicOptionsContainer",
			sections: [{
				//title: "Generic options",
				rows: [{
					orientation: "vertical",
					field: {
						name: "InterchangeSenderID",
						displayName: "Interchange Sender ID (Segment ISA, Element ISA06)",
						labelSize: "grander"
					}
				}, {
					orientation: "vertical",
					field: {
						name: "InterchangeReceiverQualifier",
						displayName: "Interchange Receiver Qualifier (Segment ISA, Element ISA07)",
						labelSize: "grander"
					}
				}, {
					orientation: "vertical",
					field: {
						name: "InterchangeReceiverID",
						displayName: "Interchange Receiver ID (Segment ISA, Element ISA08)",
						labelSize: "grander"
					}
				}, {
					field: {
						name: "RequestInterchangeAcknowledgement",
						displayName: "Request Interchange Acknowledgement? (Segment ISA, Element ISA14)",
						labelSize: "grandest"
					}
				}, {
					orientation: "vertical",
					field: {
						name: "ApplicationSenderID",
						displayName: "Application Sender ID (Segment GS, Element GS02)",
						labelSize: "grander"
					}
				}, {
					orientation: "vertical",
					field: {
						name: "ApplicationReceiverID",
						displayName: "Application Receiver ID (Segment GS, Element GS03)",
						labelSize: "grander"
					}
				}]
			}]
		}, {
			attachPoint: "claimOptionsContainer",
			sections: [{
				title: "Provider Details  (837I: Locator 1 / Loop 2010AA) (837P: Locator 33 / Loop 2010AA)",
				rows: [{
					fields: [{
						name: "ProviderDetailLocation",
						displayName: "Provider Detail Location:",
						labelSize: "larger"
					}]
				}]
			}, {
				title: "Additional Provider Details (837I: Locator 81 / Loop 2000A, Segment PRV, Element PRV03) (837P: Locator 33b / Loop 2000A, Segment PRV, Element PRV03)",
				rows: [{
					fields: [{
						name: "TaxonomyCodeQualifier",
						displayName: "Taxonomy Code Qualifier:",
						labelSize: "larger",
						fieldSize: "smaller"
					}, {
						name: "TaxonomyCode",
						displayName: "Taxonomy Code:",
						labelSize: "large"
					}]
				}, {
					fields: [{
						name: "OtherProviderIdentifier",
						displayName: "Other Provider Identifier:",
						labelSize: "larger"
					}]
				}]
			}, {
				title: "Medicaid Provider Number Details (837I: Locator 51 / Loop 2000B, SBR)",
				rows: [{
					fields: [{
						name: "IncludeMedicaidProviderNumber",
						displayName: "Use Medicaid Provider Number (MA#) from Location Table on the claim",
						labelSize: "grander",
						fieldSize: "smallest"
					}]
				}]
			}, {
				title: "Attending Provider Specialty Information (837I: Locator 81 / Loop 2310A, Segment PRV, Element PRV03)",
				rows: [{
					fields: [{
						name: "IncludeAttendingProviderTaxonomyCode",
						displayName: "Include Attending Provider Taxonomy Code",
						labelSize: "larger"
					}]
				}]
			}, {
				title: "Attending Provider Secondary Identification (837I: Locator 76 / Loop 2310A, Segment REF, Element REF02)",
				rows: [{
					fields: [{
						name: "AttendingProviderOtherId",
						displayName: "Attending Provider Secondary Identifier:",
						labelSize: "largest"
					}]
				}]
			}, {
				title: "Referring Provider (837I: Locator 78 / Loop 2310F, Segment REF, Element REF02) (837P: Locator 17a / Loop 2310A, Segment REF, Element REF02)",
				rows: [{
					fields: [{
						name: "IncludeCertifyingPhysician",
						displayName: "Use Primary on POC (Overrides 'Include Referring Physician' and 'Allow Duplicates')",
						labelSize: "large",
						containerStyle: "display: none;"
					}, {
						name: "IncludeRefPhy",
						displayName: "Include Referring Physician",
						labelSize: "large",
						containerStyle: "display: none;"
					}, {
						name: "AllowDupPhy",
						displayName: "Allow Duplicates",
						labelSize: "large",
						containerStyle: "display: none;"
					}, {
						name: "RefProviderOtherId",
						displayName: "Referring Provider Other Id:",
						labelSize: "larger"
					}]
				}]
			}, {
				title: "Service Facility (837I: Locator 78 / Loop 2310E)",
				rows: [{
					fields: [{
						name: "IncludeRoomAndBoardFacility",
						displayName: "Include Room and Board Facility",
						labelSize: "large",
						containerStyle: "display: none;"
					}]
				}]
			}, {
				title: "Type of Bill (837I: Locator 4 / Loop 2300, Segment CLM, Element CLM05) (837P: Locator 24b / Loop 2300, Segment CLM, Element CLM05)",
				containerAttachPoint: "typeOfBillContainer",
				rows: [{
					fields: [{
						name: "TypeOfCare",
						displayName: "Type of Care:",
						labelSize: "medium",
						fieldSize: "medium"
					}]
				}, {
					fields: [{
						name: "TypeOfBillOverride",
						displayName: " Type of Bill Override:",
						labelSize: "large",
						fieldSize: "medium"
					}]
				}]
			}, {
				title: "Provider Signature on File (837P: Loop 2300, Segment CLM, Element CLM06)",
				containerAttachPoint: "ProviderSignatureOnFileContainer",
				rows: [{
					fields: [{
						name: "ProviderSignature",
						displayName: "Provider Signature on File:",
						labelSize: "larger",
						fieldSize: "medium"
					}]
				},{
					fields: [{
						name: "ProviderSigOnFileText",
						displayName: "Provider Signature Text:",
						labelSize: "larger",
						fieldSize: "medium"
					}]
				},{
					fields: [{
						name: "ProviderSigOnFileDate",
						displayName: "Provider Signature Date:",
						labelSize: "larger"
					}]
				}]
			}, {
				title: "Statement Dates",
				containerAttachPoint: "claimStatementDatesContainer",
				rows: [{
					containerAttachPoint: "statementDatesRow",
					fields: [{
						name: "StatementFromDate",
						displayName: "Statement From Date:",
						labelSize: "larger"
					}, {
						name: "StatementThroughDate",
						displayName: "Statement Through Date:",
						labelSize: "larger"
					}]
				}]
			}, {
				title: "Prior Authorizations (837I: Locator 63 / Loop 2300, Segment REF[G1], Element REF02) (837P: Locator 23 / Loop 2300, Segment REF[G1], Element REF02)",
				rows: [{
					fields: [{
						name: "AllowMultiplePriorAuth",
						displayName: "Claim may contain more than 1 prior authorization number",
						labelSize: "grander",
						fieldSize: "smallest"
					}]
				 }, {
					fields: [{
						name: "TreatmentAuthorizationCode",
						displayName: "Treatment Authorization Code:",
						labelSize: "larger"
					}]
				}]
			}, {
				title: "Payer Claim Control Number (837I: Locator 64 / Loop 2300, Segment REF[F8], Element REF02, 837P: Locator 22 / Loop 2300, Segment REF[F8], Element REF02)",
				rows: [{
					fields: [{
						name: "HasDCNForReplacementClaims",
						displayName: "Include DCN for Replacement Claims?",
						fieldSize: "smallest",
						labelSize: "grandest"
					}]
				}]
			}, {
				title: "Other Payer's Claim Control Number (837I: Locator 64 / Loop 2330B, Segment REF[F8], Element REF02, 837P: Locator 22 / Loop 2330B, Segment REF[F8], Element REF02)",
				rows: [{
					fields: [{
						name: "HasDCNForTransferredCharges",
						displayName: "Include DCN for Transferred Charges?",
						fieldSize: "smallest",
						labelSize: "grandest"
					}]
				}]
			}, {
				title: "Diagnoses  (837I: Locator 66-67 / Loop 2300, Segment HI[ABK,ABF]) (837P: Locator 21 / Loop 2300, Segment HI[ABK,ABF])",
				rows: [{
					fields: [{
						name: "DiagnosesSource",
						displayName: "Diagnoses Source:",
						labelSize: "large"
					}]
				}]
			}, {
				title: "ECI Codes  (837I: Locator 72 / Loop 2300, Segment HI[BN])",
				rows: [{
					fields: [{
						name: "EciCodes",
						displayName: "ECI Codes:",
						labelSize: "large"
					}]
				}]
			}, {
				title: "Procedures (837I: Locator 74 / Loop 2300, Segment HI[BBR,BBQ])",
				rows: [{
					fields: [{
						name: "ProceduresSource",
						displayName: "Procedures Source:",
						labelSize: "large"
					}]
				}]
			}, {
				title: "Service Lines (837I: Locator 42-49 / Loop 2400, Segment SV2) (837P: Locator 24 / Loop 2400, Segment SV1)",
				rows: [{
					fields: [{
						name: "TopServiceLineWithHipps",
						displayName: "Add top service line w/ HIPPS",
						labelSize: "largest",
						fieldSize: "smallest"
					}]
				}, {
					containerAttachPoint: "topServiceLineContainer",
					fields: [{
						name: "TopServiceLineRevenueCode",
						displayName: "Revenue Code:",
						labelSize: "medium-large",
						fieldSize: "small"
					}, {
						name: "TopServiceLineQuantity",
						displayName: "Quantity:",
						labelSize: "medium-small",
						fieldSize: "small"
					}, {
						name: "TopServiceLineChargeAmount",
						displayName: "Charge Amount:",
						labelSize: "medium-large",
						fieldSize: "small"
					}]
				}, {
					fields: [{
						name: "WeeklyReportingOfGIP",
						displayName: "Visit Reporting: Weekly reporting of GIP while in Q5006",
						labelSize: "grander",
						fieldSize: "smallest"
					}]
				}, {
					fields: [{
						name: "RevenueCodeIncludeLeadingZero",
						displayName: "Revenue Code Include Leading Zero",
						labelSize: "largest",
						fieldSize: "smallest"
					}]
				}, {
					fields: [{
						name: "CombineMedicationCharges",
						displayName: "Combine medication charges into single service line per revenue code",
						labelSize: "large",
						fieldSize: "small"
					}]
				}, {
					fields: [{
						name: "CombineSupplyCharges",
						displayName: "Combine supply charges into single service line per revenue code",
						labelSize: "large",
						fieldSize: "small"
					}]
				}, {
					fields: [{
						name: "DoNotCombineLOC",
						displayName: "Do not combine level of care charges",
						labelSize: "large",
						fieldSize: "small"
					}]
				}, {
					fields: [{
						name: "CombineVisitByDateRevenueCode",
						displayName: "Combine visits by date and revenue code",
						labelSize: "large",
						fieldSize: "small"
					}]
				}, {
					fields: [{
						name: "CombineRoomAndBoard",
						displayName: "Combine Room and Board charges into single service line per revenue code",
						labelSize: "large",
						fieldSize: "small"
					}]
				}, {
					fields: [{
						name: "CopayAsAmountPaid",
						displayName: "Use insurance copay as Patient Amount Paid in AMT*F5",
						fieldSize: "smallest",
						labelSize: "grandest"
					}]
				}, {
					fields: [{
						name: "DoNotOverwriteHCPCS",
						displayName: "Do not overwrite HCPCS code with Q code in Level Of Care charges",
						labelSize: "large",
						fieldSize: "small"
					}]
				}, {
					fields: [{
						name: "requireOrderingProvider",
						displayName: "Ordering Provider required in loop2420E",
						labelSize: "large",
						fieldSize: "small"
					}]
				}, {
					fields: [{
						name: "ExcludeChargeTypes",
						displayName: "Exclude Charges",
						labelSize: "large",
						fieldSize: "small"
					}]
				}, {
					fields: [{
						name: "ServiceUnitCount",
						displayName: "Service Unit Count:",
						labelSize: "large"
					}, {
						name: "ServiceUnitsPerCharge",
						displayName: "Units Per Charge:",
						labelSize: "large",
						fieldSize: "small"
					}]
				}, {
					containerAttachPoint: "serviceUnitCountGridContainer"
				}, {
					fields: [{
						name: "ChargeAmount",
						displayName: "Charge Amount:",
						labelSize: "large"
					}]
				}, {
					fields: [{
						name: "DefaultHippsCode",
						displayName: "Default HIPPS Code (PDGM):",
						labelSize: "large",
						fieldSize: "small"
					}]
				}, {	
					fields: [{
						name: "DefaultNpiSource",
						displayName: "Default NPI Source:",
						labelSize: "large",
						fieldSize: "small"
					}]
				}, {
					field: {
						name: "CustomNpiSourcesLabel",
						displayName: "Custom NPI Source(s)",
						fieldSize: "small",
						labelSize: "grand",
						hideLabel: false,
                        containerStyle: "",
						noField: true
					}
				}, {
					containerAttachPoint: "customNpiSourceGridContainer"
				}]
			}]
		}, {
			attachPoint: "eligibilityContainer",
			sections: [{
				//title: "",
				rows: [{
					orientation: "vertical",
					field: {
						name: "CustomInformationSourceName",
						displayName: "Custom Information Source Name (Loop 2100A, Segment NM1, Element NM103)",
						labelSize: "grandest"
					}
				}, {
					orientation: "vertical",
					field: {
						name: "InformationSourceIdQualifier",
						displayName: "Information Source ID Qualifier (Loop 2100A, Segment NM1, Element NM108)",
						labelSize: "grandest"
					}
				}, {
					orientation: "vertical",
					field: {
						name: "CustomInformationSourceId",
						displayName: "Custom Information Source ID (Loop 2100A, Segment NM1, Element NM109)",
						labelSize: "grandest"
					}
				}, {
					orientation: "vertical",
					field: {
						name: "InformationReceiverIdQualifier",
						displayName: "Information Receiver ID Qualifier (Loop 2100B, Segment NM1, Element NM108)",
						labelSize: "grandest"
					}
				}, {
					orientation: "vertical",
					field: {
						name: "CustomInformationReceiverID",
						displayName: "Custom Information Receiver ID (Loop 2100B, Segment NM1, Element NM109)",
						labelSize: "grandest"
					}
				}, {
					orientation: "vertical",
					field: {
						name: "InformationReceiverIndividualOrOrganizationName",
						displayName: "Information Receiver Name (Loop 2100B, Segment NM1, Element NM103)",
						labelSize: "grandest"
					}
				}, {
					orientation: "vertical",
					field: {
						name: "InformationReceiverIdentificationCodeQualifier",
						displayName: "Information Receiver Identification Code Qualifier (Loop 2100B, Segment NM1, Element NM108)",
						labelSize: "grandest"
					}
				}, {
					orientation: "vertical",
					field: {
						name: "InformationReceiverIdentificationCode",
						displayName: "Information Receiver Identification Code (Loop 2100B, Segment NM1, Element NM109)",
						labelSize: "grandest"
					}
				}, {
					field: {
						name: "IncludeTaxonomyCode",
						displayName: "Include Taxonomy Code? (Loop 2100B, Segment PRV)",
						labelSize: "grandest"
					}
				}, {
					orientation: "vertical",
					field: {
						name: "CustomTaxonomyCode",
						displayName: "Custom Taxonomy Code (Loop 2100B, Segment PRV, Element PRV03)",
						labelSize: "grandest"
					}
				}, {
					orientation: "vertical",
					field: {
						name: "ServiceTypeCodes",
						displayName: "Service Type Codes (Loop 2110C, Segment EQ, Element EQ01)",
						labelSize: "grandest"
					}
				}]
			}]
		}, {
			attachPoint: "CurrentIllnessContainer",
			sections: [{
				title: "Date Of Current Illness (837P: Locator 14 / Loop 2300, Segment DTP, Element DTP03)",
				rows: [{
					fields: [{
						name: "CurrentIllnessDate",
						displayName: "Current Illness Date:",
						labelSize: "larger"
					}]
				}]
			}]
		}, {
			attachPoint: "IncludeInsContainer",
			sections: [{
				title: "Subscriber Information (Loop 2000B, Segment SBR)",
				rows: [{
					fields: [{
						name: "includeRelationship",
						displayName: "Include Relationship",
						labelSize: "large",
						fieldSize: "small"
					}]
				}]
			}]
					
		}, {
			attachPoint: "RequireCoordOfBenefitsContainer",
			sections: [{
				title: "Other Subscriber Information (Loop 2320, Segment SBR)",
				rows: [{
					fields: [{
						name: "RequireCOB",
						displayName: "Include other Coordination Of Benefits payers",
						labelSize: "large",
						fieldSize: "small"
					}]
				}]
			}]
					
		}, {			
			attachPoint: "patientStatementContainer",
			sections: [{
				title: "Statement Dates",
				containerAttachPoint: "patientStatementDatesContainer"
			}, {
				title: "Pay To",
				rows: [{
					fields: [{
						name: "PayToName",
						displayName: "Name",
						fieldSize: "grandest",
						labelSize: "smaller"
					}]
				}, {
					fields: [{
						name: "PayToAddressStreet1",
						displayName: "Street",
						labelSize: "smaller"
					}]
				}, {
					fields: [{
						name: "PayToAddressStreet2",
						displayName: "",
						labelSize: "smaller"
					}]
				}, {
					fields: [{
						name: "PayToAddressCity",
						displayName: "City",
						labelSize: "smaller"
					}]
				}, {
					fields: [{
						name: "PayToAddressState",
						displayName: "State",
						fieldSize: "small",
						labelSize: "smaller"
					}, {
						name: "PayToAddressZip",
						displayName: "Zip",
						fieldSize: "small",
						labelSize: "smallest"
					}]
				}, {
					fields: [{
						name: "PaymentPhoneNumber",
						displayName: "Payment Phone Number",
						labelSize: "medium-large"
					}]
				}, {
					fields: [{
						name: "PaymentWebsite",
						displayName: "Payment Website",
						fieldSize: "grandest",
						labelSize: "medium-large"
					}]
				}, {
					fields: [{
						name: "PaymentDueDate",
						displayName: "Payment Due",
						labelSize: "medium-large"
					}]
				}]
			}, {
				title: "Provider Details",
				rows: [{
					fields: [{
						name: "PatientStatementProviderDetailLocation",
						displayName: "Provider Detail Location:",
						labelSize: "larger"
					}]
				}, {
					fields: [{
						name: "PatientStatementLogoPath",
						displayName: "Logo Path:"
					}]
				}]
			}]
		}, {
			attachPoint: "additionalCodingContainer",
			sections: [{
				title: "Additional Coding",
				hideDivider: false,
				rows: [{
					fields: [{
						name: "Scenario1",
						displayName: "Include Occurrence Code 61 (or 62) w/ Inpatient Discharge Date when inpatient discharge has occurred within 14 days of bill period start",
						fieldSize: "smallest",
						labelSize: "grandest"
					}]
				}, {
					fields: [{
						name: "Scenario8",
						displayName: "Include CBSA in value code(s) 61 and/or G8 based on Level of Care provided to patient",
						fieldSize: "smallest",
						labelSize: "grandest"
					}]
				}, {
					fields: [{
						name: "Scenario2",
						displayName: "Include Occurrence Span Code 77 w/ Uncovered Date Range when NOE was submitted late",
						fieldSize: "smallest",
						labelSize: "grandest"
					}]
				}, {
					fields: [{
						name: "Scenario3",
						displayName: "Include Occurrence Span Code M2 w/ Respite Care Date Range for each set Respite Care days when multiple sets of respite days are included on the claim",
						fieldSize: "smallest",
						labelSize: "grandest"
					}]
				}, {
					fields: [{
						name: "Scenario4",
						displayName: "Include Occurrence Code 42 w/ Discharge Date when Discharge Reason has an HIS Reason for Discharge that is \"02 - Revoked\"",
						fieldSize: "smallest",
						labelSize: "grandest"
					}]
				}, {
					fields: [{
						name: "Scenario5",
						displayName: "Include Condition Code 52 when Discharge Reason has an HIS Reason for Discharge that is \"04 - Moved out of hospice service area\"",
						fieldSize: "smallest",
						labelSize: "grandest"
					}]
				}, {
					fields: [{
						name: "Scenario6",
						displayName: "Include Condition Code H2 when Discharge Reason has an HIS Reason for Discharge that is \"06 - Discharge for cause\"",
						fieldSize: "smallest",
						labelSize: "grandest"
					}]
				}, {
					fields: [{
						name: "Scenario7",
						displayName: "Include Remark w/ Discharge Reason & Date when Discharge Reason has an HIS Reason for Discharge that is NOT \"01 - Expired\"",
						fieldSize: "smallest",
						labelSize: "grandest"
					}]
				}, {
					fields: [{
						name: "Scenario10",
						displayName: "Include Condition Code 47 when B-Transfer from another HHA selected as Source of Admission",
						fieldSize: "smallest",
						labelSize: "grandest"
					}]
				}, {
					containerAttachPoint: "additionalCodingGridContainer"
				}]
			}]
		}]
	};

	ClaimMapping = claimMappingModel.getObject();
	var ClaimMappingEditor = declare([ObjectEditor], {
		objectModel: claimMappingModel,
		templateString: template.generate(claimMappingTemplate),
		requirementCollections: [],
		documentReqCollections: [],

		postMixInProperties: function() {
			var t = this;
			t.inherited("postMixInProperties", arguments);

			t.getPropertyModel("PaymentDueDate").getField = function(node) {
				return new NumberTextBox(this.getFieldProperties(), node);
			};
			t.getPropertyModel("PaymentDueDate").getInternal = function(value) {
				return (value ? value : (value === 0 ? value : ""));
			};

			t.getPropertyModel("TopServiceLineQuantity").getField = function(node) {
				return new NumberTextBox(lang.mixin(this.getFieldProperties(), {constraints: {places: "0", min: 0, max: 99}}), node);
			};
			t.getPropertyModel("TopServiceLineQuantity").getInternal = function(value) {
				return (value ? value : (value === 0 ? value : ""));
			};

			t.getPropertyModel("TopServiceLineChargeAmount").getField = function(node) {
				return new NumberTextBox(lang.mixin(this.getFieldProperties(), {constraints: {places: "0,2"}}), node);
			};
			t.getPropertyModel("TopServiceLineChargeAmount").getInternal = function(value) {
				return (value ? value : (value === 0 ? value : ""));
			};
			
			t.getPropertyModel("ServiceUnitsPerCharge").getField = function(node) {
				return new NumberTextBox(lang.mixin(this.getFieldProperties(), {constraints: {places: "0", min: 0, max: 999}}), node);
			};
			t.getPropertyModel("ServiceUnitsPerCharge").getInternal = function(value) {
				return (value ? value : (value === 0 ? value : ""));
			};

			// Override default behavior to allow carets in this field
			t.getPropertyModel("ServiceTypeCodes").getInternal = function(value) {
				var t = this;
				return (t._inLoad ? value : value.toString().replace(/(~|`)/g,""));
			};

			t.getPropertyModel("InsuranceType").noDefault = true;
		},

		buildRendering: function() {
			var t = this;
			t.inherited("buildRendering", arguments);

			t.Insurances = new MultiSelect({ style: "min-width: 300px;" }, t.InsurancesFieldContainer);
			
			t.copyClaimMappingButton = new Button({
				label: "Copy Claim Mapping",
				onClick: function() {
					var saveData = t._getSaveData();
					t.pageEditor.object.set("editMode", "add");
					delete saveData.Name;
					delete saveData.id;
					t.set("object", new ClaimMapping(saveData));
				}
			},t.copyButtonLabelContainer);
			
			if (t.InsurancesContainer.parentNode) {
				t.InsurancesContainer.parentNode.style.display = 'block';
				t.InsurancesContainer.style.display = 'inline-block'
				t.copyButtonContainer.style.display = 'inline-block'
			}
			
			domStyle.set(t.InterchangeSenderIDField.domNode, "width", "140px");
			domStyle.set(t.InterchangeReceiverQualifierField.domNode, "width", "50px");
			domStyle.set(t.InterchangeReceiverIDField.domNode, "width", "140px");
			domStyle.set(t.ApplicationSenderIDField.domNode, "width", "140px");
			domStyle.set(t.ApplicationReceiverIDField.domNode, "width", "140px");
			domStyle.set(t.TaxonomyCodeQualifierField.domNode, "width", "50px");
			domStyle.set(t.TaxonomyCodeField.domNode, "width", "110px");
			domStyle.set(t.TypeOfBillOverrideField.domNode, "width", "50px");
			domStyle.set(t.PayToAddressStateField.domNode, "width", "50px");
			domStyle.set(t.PayToAddressZipField.domNode, "width", "100px");
			//domStyle.set(t.PayToNameField.domNode, "width", "250px");
			//domStyle.set(t.PaymentWebsiteField.domNode, "width", "250px"); 
			domStyle.set(t.PaymentDueDateField.domNode, "width", "50px");
			domStyle.set(t.TopServiceLineRevenueCodeField.domNode, "width", "50px");
			domStyle.set(t.TopServiceLineQuantityField.domNode, "width", "60px");
			domStyle.set(t.TopServiceLineChargeAmountField.domNode, "width", "60px");
			domStyle.set(t.topServiceLineContainer,"margin-left","40px");
			domStyle.set(t.DefaultHippsCodeField.domNode, "width", "60px");
			domStyle.set(t.ServiceUnitsPerChargeField.domNode, "width", "60px");
			
			// Move checkboxes before labels
			arrayUtils.forEach(t.objectModel.getPropertyModels(),function(propertyModel) {
				if ((propertyModel.type === "boolean") && (propertyModel.name != "BackgroundImagePathAlt") && (propertyModel.name != "PrintWithFormImage")) {
					var propName = propertyModel.name;
					
					var fieldContainer = t[propName + "FieldContainer"];
					var labelContainer = t[propName + "LabelContainer"];
					
					domStyle.set(fieldContainer, "min-width", "20px");
					domStyle.set(labelContainer, "width", "725px");

					domConstruct.place(fieldContainer, labelContainer, "before");
				}
			});

			domConstruct.create("label", { innerHTML: "&nbsp;days from Process Date" }, t.PaymentDueDateFieldContainer);
			
			t.serviceUnitCountGrid = new ServiceUnitCountGrid();
			t.serviceUnitCountGrid.startup();
			domConstruct.place(t.serviceUnitCountGrid.domNode, t.serviceUnitCountGridContainer, "last");
			
			t.customNpiSourceGrid = new CustomNpiSourceGrid();
			t.customNpiSourceGrid.startup();
			domConstruct.place(t.customNpiSourceGrid.domNode, t.customNpiSourceGridContainer, "last");

			t.additionalCodingGrid = new AdditionalCodeGrid();
			t.additionalCodingGrid.startup();
			domConstruct.place(t.additionalCodingGrid.domNode, t.additionalCodingGridContainer, "last");
		},

		postCreate: function() {
			var t = this;
			t.inherited(arguments);

			t.TaxonomyCodeQualifierField.set({ pattern: "[A-Za-z0-9]{2}" });
			t.InterchangeReceiverQualifierField.set({ pattern: "[A-Za-z0-9]{2}" });
			t.TaxonomyCodeField.set({ pattern: "[A-Za-z0-9]{10}" });
			t.PaymentDueDateField.set("constraints", { fractional: false, min: 0 });
			t.PayToAddressStateField.set("pattern", "[A-Z]{2}");
			t.PayToAddressStateField.set("uppercase", true);
			t.PayToAddressZipField.set("pattern", "\\d{5}(-\\d{4})?");
			t.PaymentPhoneNumberField.set("pattern", "(\\d{3}-)?\\d{3}-\\d{4}");
			t.PaymentPhoneNumberField.set("invalidMessage", "123-4567 or 123-456-7890 format");
			t.TopServiceLineRevenueCodeField.set({ regExp: revenueCodeRegEx, invalidMessage: revenueCodeInvalidMessage });
			t.ServiceTypeCodesField.set({pattern: "([A-Za-z0-9]{1,2})(\\^([A-Za-z0-9]{1,2})){0,98}",
										 invalidMessage: 'Enter the desired Service Type Code(s), separated by "^"' });
			t.DefaultHippsCodeField.set("uppercase", true);
			t.DefaultHippsCodeField.set({ pattern: "([A-Za-z0-9]{1}\\d{4}|\\d{1}[A-Za-z]{2}\\d{2}|\\d{1}[A-Za-z]{3}[A-Za-z0-9]{1}|\\d{4})" });
			t.TypeOfBillOverrideField.set({pattern: "\\d{3}", invalidMessage: "Use 3 digits"});

			t.own(aspect.before(t, "_setObjectAttr", function(object) {
				if (!object.ProviderDetailLocation) {
					// default to Location
					object.ProviderDetailLocation = 3;
				}
				if (!object.AllowMultiplePriorAuth) {
					// default to true
					object.AllowMultiplePriorAuth = true;
				}
				if (!object.IncludeMedicaidProviderNumber) {
					// default to false
					object.IncludeMedicaidProviderNumber = false;
				}
				if (!object.TreatmentAuthorizationCode) {
					// default to Authorization #
					object.TreatmentAuthorizationCode = 3;
				}
				if (!object.DiagnosesSource) {
					// default to POC / POF
					object.DiagnosesSource = 1;
				}
				if (!object.EciCodes) {
					// default to include as DX codes
					object.EciCodes = 1;
				}
				if (!object.ProceduresSource) {
					// default to POC / POF
					object.ProceduresSource = 1; 
				}
				if (!object.RevenueCodeIncludeLeadingZero) {
					// default to true
					object.RevenueCodeIncludeLeadingZero = true;
				}
				if (!object.CombineMedicationCharges) {
					// default to false
					object.CombineMedicationCharges = false;
				}
				if (!object.CombineSupplyCharges) {
					// default to false
					object.CombineSupplyCharges = false;
				}
				if (!object.DoNotCombineLOC) {
					// default to false
					object.DoNotCombineLOC = false;
				}
				if (!object.CombineVisitByDateRevenueCode) {
					// default to false
					object.CombineVisitByDateRevenueCode = false;
				}
				if (!object.CombineRoomAndBoard) {
					// default to false
					object.CombineRoomAndBoard = false;
				}
				if (!object.requireOrderingProvider) {
					// default to false
					object.requireOrderingProvider = false;
				}
				if (!object.RequireCOB) {
					// default to false
					object.RequireCOB = false;
				}
				if (!object.includeRelationship) {
					// default to false
					object.includeRelationship = false;
				}
				if (!object.ServiceUnitCount) {
					// default to Per Charge
					object.ServiceUnitCount = 1; 
					object.ServiceUnitsPerCharge = 1;
				}
				if (!object.ChargeAmount) {
					// default to Gross Rate
					object.ChargeAmount = 1; 
				}
				if (!object.InsuranceType) {
					object.InsuranceType = 1; // default to Homecare
				}
				if (!object.HasDCNForReplacementClaims) {
					// default to true
					object.HasDCNForReplacementClaims = true;
				}
				if (!object.HasDCNForTransferredCharges) {
					// default to false
					object.HasDCNForTransferredCharges = false;
				}
				if (!object.DoNotOverwriteHCPCS) {
					// default to false
					object.DoNotOverwriteHCPCS = false;
				}
			}));

			t.own(aspect.after(t, "_setObjectAttr", function(object) {
				t.additionalCodingGrid.set("object", object.AdditionalCodes);
				t.additionalCodingGrid.refreshGridBody();

				t.serviceUnitCountGrid.set("object", object.ServiceUnitCounts);
				t.serviceUnitCountGrid.refreshGridBody();

				t.customNpiSourceGrid.set("object", object.CustomNpiSources);
				t.customNpiSourceGrid.refreshGridBody();

				// Populate Insurances with those that use documentation requirements with this claim mapping
				query("option", t.Insurances.containerNode).forEach(function(option) {domConstruct.destroy(option);});
				TableManager.get("MappingByInsurance").query({
					mapping: object.id
				}).forEach(function(mapping) {
					domConstruct.create("option", {
						label:escapeText(mapping.name), id:mapping.id, disabled: true
					}, t.Insurances.containerNode);
				});
			}, true));

			t.registerChangeLogic("InsuranceType", function(value) {
				if (value != 2) {
					t.object.set("WeeklyReportingOfGIP", false);
				} else if (!t.inObjectLoad) {
					// Default to Yes
					t.object.set("WeeklyReportingOfGIP", true);
				}

				var displayHospice = (value == 2) ? "" : "none";
				var displayHomecare = (value == 1) ? "" : "none";
				domStyle.set(t.WeeklyReportingOfGIPContainer, "display", displayHospice);
				domStyle.set(t.Scenario1Container, "display", displayHomecare);
				domStyle.set(t.Scenario2Container, "display", displayHospice);
				domStyle.set(t.Scenario3Container, "display", displayHospice);
				domStyle.set(t.Scenario4Container, "display", displayHospice);
				domStyle.set(t.Scenario5Container, "display", displayHospice);
				domStyle.set(t.Scenario6Container, "display", displayHospice);
				domStyle.set(t.Scenario7Container, "display", displayHospice);
				domStyle.set(t.Scenario8Container, "display", displayHospice);
				//domStyle.set(t.Scenario10Container, "display", displayHospice);
				//domStyle.set(t.typeOfBillContainer, "display", displayHospice);
				domStyle.set(t.TypeOfCareContainer, "display", displayHospice);
				domStyle.set(t.CombineMedicationChargesContainer, "display", displayHospice);
				domStyle.set(t.IncludeCertifyingPhysicianContainer, "display", displayHospice);

				t.require("TypeOfCare", (domStyle.get(t.claimOptionsContainer, "display") == "") ? false : true);

				if (value == 1) {
					t.object.set("Scenario2", false);
					t.object.set("Scenario3", false);
					t.object.set("Scenario4", false);
					t.object.set("Scenario5", false);
					t.object.set("Scenario6", false);
					t.object.set("Scenario7", false);
					t.object.set("Scenario8", false);
					//t.object.set("Scenario10", false);
					t.object.set("TypeOfCare", "");
					t.object.set("CombineMedicationCharges", false);
				} else if (value == 2) {
					t.object.set("Scenario1", false);
				}

				// Filter claim form by insurance type. No insurance type on the form means it's unrestricted
				var restrictIdArray = [];
				TableManager.get("Tables.Insurance.ClaimForm").query().forEach(function(claimForm) {
					var insuranceType = claimForm.InsuranceType;
					if (insuranceType == value || insuranceType == "" || value == "") {
						restrictIdArray.push(claimForm.id);
					}
				});

				// Remove current claim form selection if invalid
				var newClaimFormVal = t.object.get("ClaimForm");
				if (arrayUtils.indexOf(restrictIdArray, newClaimFormVal) == -1) {
					newClaimFormVal = "";
				}
				t.populateSelect(t.ClaimFormField, TableManager.get("Tables.Insurance.ClaimForm"),
								 newClaimFormVal, true, restrictIdArray);
				t.object.set("ClaimForm", newClaimFormVal);

				t.showTopServiceLineFields();
			});

			t.registerChangeLogic("ClaimForm", function(value) {
				var claimForm = {};
				if (value) {
					claimForm = TableManager.get("Tables.Insurance.ClaimForm").get(value);
				}
				
				if (!claimForm.AllowFormImage) {
					t.object.set("PrintWithFormImage", "");
					t.object.set("BackgroundImagePathAlt", "");
				}
				
				domStyle.set(t.PrintWithFormImageContainer, "display", claimForm.AllowFormImage ? "" : "none");
				domStyle.set(t.BackgroundImagePathAltContainer, "display", claimForm.AllowFormImage ? "" : "none");

				var showClaimFormOptions = (claimForm.Type == "UB-04/CMS-1450" || claimForm.Type == "CMS-1500" ||
											claimForm.Type == "837I" || claimForm.Type == "837P");

				domStyle.set(t.claimOptionsContainer, "display", showClaimFormOptions ? "" : "none");
				domStyle.set(t.genericElectronicOptionsContainer, "display", claimForm.Electronic ? "" : "none");
				domStyle.set(t.additionalCodingContainer, "display", ((claimForm.Type != undefined) && (claimForm.Type != "270") && (claimForm.Type != "patientStatement")) ? "" : "none");
				domStyle.set(t.patientStatementContainer, "display", (claimForm.Type == "patientStatement") ? "" : "none");
				domStyle.set(t.eligibilityContainer, "display", (claimForm.Type == "270") ? "" : "none");
				domStyle.set(t.DefaultHippsCodeContainer, "display", (claimForm.name == "Electronic RAP Claim (837I)" ? "" : "none"));
				domStyle.set(t.ProviderSignatureOnFileContainer, "display", (claimForm.Type == "CMS-1500" || claimForm.Type == "837P" ? "" : "none"));
				domStyle.set(t.ProviderSignatureContainer, "display", (claimForm.Type == "CMS-1500" || claimForm.Type == "837P" ? "" : "none"));
				domStyle.set(t.ProviderSigOnFileTextContainer, "display", (claimForm.Type == "CMS-1500" ? "" : "none"));
				domStyle.set(t.ProviderSigOnFileDateContainer, "display", (claimForm.Type == "CMS-1500" ? "" : "none"));
				domStyle.set(t.DefaultNpiSourceContainer, "display", (claimForm.Type == "CMS-1500" ? "" : "none"));
				domStyle.set(t.CustomNpiSourcesLabelContainer, "display", (claimForm.Type == "CMS-1500" ? "" : "none"));
				domStyle.set(t.customNpiSourceGridContainer, "display", (claimForm.Type == "CMS-1500" ? "" : "none"));
				domStyle.set(t.OtherProviderIdentifierContainer, "display", (claimForm.Type == "CMS-1500" ? "" : "none"));
				domStyle.set(t.CopayAsAmountPaidContainer, "display", (claimForm.Type == "837P" ? "" : "none"));
				domStyle.set(t.IncludeCertifyingPhysicianContainer, "display", ((t.object.get("InsuranceType") == 2) && (claimForm.Type == "837I" || claimForm.Type == "UB-04/CMS-1450") ? "" : "none")); // Show for hospice
				domStyle.set(t.IncludeAttendingProviderTaxonomyCodeContainer, "display", (claimForm.Type == "837I" || claimForm.Type == "UB-04/CMS-1450" ? "" : "none"));
				domStyle.set(t.AttendingProviderOtherIdContainer, "display", (claimForm.Type == "837I" || claimForm.Type == "UB-04/CMS-1450" ? "" : "none"));
				domStyle.set(t.IncludeRefPhyContainer, "display", (claimForm.Type == "CMS-1500" || claimForm.Type == "837P" || claimForm.Type == "837I" || claimForm.Type == "UB-04/CMS-1450" ? "" : "none"));
				domStyle.set(t.RefProviderOtherIdContainer, "display", (claimForm.Type == "CMS-1500" || claimForm.Type == "837P" || claimForm.Type == "837I" || claimForm.Type == "UB-04/CMS-1450" ? "" : "none"));
				domStyle.set(t.AllowDupPhyContainer, "display", (claimForm.Type == "CMS-1500" || claimForm.Type == "837P" || claimForm.Type == "837I" || claimForm.Type == "UB-04/CMS-1450" ? "" : "none"));
				domStyle.set(t.IncludeRoomAndBoardFacilityContainer, "display", (claimForm.Type == "837I" || claimForm.Type == "UB-04/CMS-1450" ? "" : "none"));
				domStyle.set(t.CurrentIllnessContainer, "display", (claimForm.Type == "CMS-1500" || claimForm.Type == "837P"? "" : "none"));
				domStyle.set(t.requireOrderingProviderContainer, "display", (claimForm.Type == "837P"?"": "none"));	
				domStyle.set(t.RequireCoordOfBenefitsContainer, "display", (true?"": "none"));	
				domStyle.set(t.IncludeInsContainer, "display", ((claimForm.Type == "CMS-1500" || claimForm.Type == "UB-04/CMS-1450" || claimForm.Type == "patientStatement")?"" : "none"));
				domStyle.set(t.CombineSupplyChargesContainer, "display", (claimForm.Type == "CMS-1500" ? "none" : ""));
				domStyle.set(t.DoNotCombineLOCContainer, "display", "");
				domStyle.set(t.CombineVisitByDateRevenueCodeContainer, "display", "");
				domStyle.set(t.CombineRoomAndBoardContainer, "display", "");
				domStyle.set(t.DoNotOverwriteHCPCSContainer, "display", "");
				domStyle.set(t.ExcludeChargeTypesContainer, "display", "");
				domStyle.set(t.TypeOfBillOverrideContainer, "display", ((claimForm.id >= 1 && claimForm.id <= 8) ? "" : "none")); // Show for actual claims only
				domStyle.set(t.CreateCopyOfClaimForInsuranceContainer, "display", ((t.object.get("InsuranceType") == 2) && (claimForm.id == 1 || claimForm.id == 2 || claimForm.id == 5 || claimForm.id == 6) ? "" : "none")); // Show for hospice claims

				t.require("ProviderDetailLocation", showClaimFormOptions);
				t.require("TypeOfCare", (showClaimFormOptions && (t.object.get("InsuranceType") == 2)));
				t.require("ProviderSignature", (claimForm.Type == "837P" || claimForm.Type == "CMS-1500"));
				t.require("ProviderSigOnFileText", (claimForm.Type == "CMS-1500"));
				t.require("ProviderSigOnFileDate", (claimForm.Type == "CMS-1500"));
				t.require("DefaultNpiSourceContainer", (claimForm.Type == "CMS-1500"));
				t.require("StatementFromDate", showClaimFormOptions);
				t.require("StatementThroughDate", showClaimFormOptions);
				t.require("TreatmentAuthorizationCode", showClaimFormOptions);
				t.require("DiagnosesSource", showClaimFormOptions);
				t.require("EciCodes", showClaimFormOptions);
				t.require("ProceduresSource", showClaimFormOptions);

				t.require("InterchangeSenderID", claimForm.Electronic);
				t.require("InterchangeReceiverQualifier", claimForm.Electronic);
				t.require("InterchangeReceiverID", claimForm.Electronic);

				t.require("PatientStatementProviderDetailLocation", (claimForm.Type == "patientStatement"));

				// Move statement date fields to appropriate container
				if (claimForm.Type == "patientStatement") {
					domConstruct.place(t.statementDatesRow, t.patientStatementDatesContainer);
				} else {
					domConstruct.place(t.statementDatesRow, t.claimStatementDatesContainer);
				}

				if (!showClaimFormOptions) {
					t.resetClaimOptions();
				}
				if (!claimForm.Electronic) {
					t.resetGeneralElectronicOptions();
				}
				if ((claimForm.Type == undefined) || (claimForm.Type == "270") || (claimForm.Type == "patientStatement")) {
					t.resetAdditionalCodingOptions();
				}
				if (claimForm.Type != "patientStatement") {
					t.resetPatientStatementOptions();
				}
				if (claimForm.Type != "270") {
					t.resetEligibilityOptions();
				}
				if (claimForm.name != "Electronic RAP Claim (837I)") {
					t.object.set("DefaultHippsCode", "");
				}
				if (claimForm.Type != "837P") {
					t.object.set("CopayAsAmountPaid", false);
					t.object.set("requireOrderingProvider", false);
				}
				if (claimForm.Type != "CMS-1500") {
					t.object.set("ProviderSigOnFileText", "");
					t.object.set("ProviderSigOnFileDate", "");
					t.object.set("DefaultNpiSource", "");
					t.object.set("OtherProviderIdentifier", "");
				} else {
					t.object.set("CombineSupplyCharges", false);
				}
				if (claimForm.Type != "CMS-1500" && claimForm.Type != "837P") {
					t.object.set("ProviderSignature", "");
					t.object.set("CurrentIllnessDate", "");
				}
				if ((claimForm.Type != "CMS-1500") && (claimForm.Type != "UB-04/CMS-1450") && (claimForm.Type != "patientStatement")) {
					t.object.set("includeRelationship", false);
				}
				if (!showClaimFormOptions && claimForm.Type != "patientStatement") {
					t.object.set("StatementFromDate", "");
					t.object.set("StatementThroughDate", "");
				}
				
				if (claimForm.Type != "UB-04/CMS-1450" && claimForm.Type != "837I") {
					t.object.set("IncludeRoomAndBoardFacility", "");
				}

				if (claimForm.id > 8) {
					t.object.set("TypeOfBillOverride", "");
				}

				if ((t.object.get("InsuranceType") != 2) || (claimForm.id != 1 && claimForm.id != 2 && claimForm.id != 5 && claimForm.id != 6)) {
					t.object.set("CreateCopyOfClaimForInsurance", "");
				}
				
				var restrictIdArray = [1,2];
				if (claimForm.id === 10 || claimForm.id === 14) {
					// Only show NOE Start option for NOTR
					restrictIdArray.push(3);
				}
				
				// Remove current statement from date selection if invalid
				var newStatementFromVal = t.object.get("StatementFromDate");
				if (arrayUtils.indexOf(restrictIdArray, newStatementFromVal) == -1) {
					newStatementFromVal = "";
				}
				t.populateSelect(t.StatementFromDateField, TableManager.get("FillInGroup~20614"),
								 newStatementFromVal, true, restrictIdArray);
				t.object.set("StatementFromDate", newStatementFromVal);

				t.showTopServiceLineFields();
				t.showAllowDupPhyField();

				// Allow only Hospice insurances in the CreateCopyOfClaimForInsurance field
				var restrictIdArray = [];
				TableManager.get("Tables.Insurance").query().forEach(function(insurance) {
					if (insurance.InsuranceType == 2) {
						restrictIdArray.push(insurance.id);
					}
				});
				t.populateSelect(t.CreateCopyOfClaimForInsuranceField, TableManager.get("Tables.Insurance"), t.object.get("CreateCopyOfClaimForInsurance"), true, restrictIdArray);
			});
			
			t.registerChangeLogic("IncludeRefPhy", function(value) {
				t.showAllowDupPhyField();
			});

			t.registerChangeLogic("TopServiceLineWithHipps", function(value) {
				domStyle.set(t.topServiceLineContainer, "display", value ? "" : "none");
				t.require("TopServiceLineRevenueCode", value);
				t.require("TopServiceLineQuantity", value);
				t.require("TopServiceLineChargeAmount", value);
				if (t.object && !value) {
					t.object.set("TopServiceLineRevenueCode","");
					t.object.set("TopServiceLineQuantity","");
					t.object.set("TopServiceLineChargeAmount","");
				}
			});

			t.registerChangeLogic("ServiceUnitCount", function(value) {
				domStyle.set(t.ServiceUnitsPerChargeContainer, "display", value == 1 ? "" : "none");
				domStyle.set(t.serviceUnitCountGridContainer, "display", value == 5 ? "" : "none");
				t.require("ServiceUnitsPerCharge", value == 1);
				t.object.set("ServiceUnitsPerCharge", value == 1 ? 1 : "");
				if (value != 5) {
					t.serviceUnitCountGrid.set("object", []);
				}
			});
		},

		showTopServiceLineFields: function() {
			var t = this;
			var show = t.object && t.object.InsuranceType == 1 && arrayUtils.indexOf([1,2,5,6],t.object.ClaimForm) >= 0;
			domStyle.set(t.TopServiceLineWithHippsContainer, "display", (show ? "" : "none"));
			if (!show) {
				t.object.set("TopServiceLineWithHipps", false);
			}
		},
		
		showAllowDupPhyField: function() {
			var t = this;
			var show = t.object && t.object.IncludeRefPhy == 1;
			domStyle.set(t.AllowDupPhyContainer, "display", (show ? "" : "none"));
			if (!show) {
				t.object.set("AllowDupPhy", false);
			}
		},

		_setActionTypeAttr: function(value) {
			/*
				value = "Edit", "Add", or "View"
			*/
			
			var t = this;
			t.inherited(arguments);

			t.copyClaimMappingButton.set("disabled", (value != "Edit"));
			t.additionalCodingGrid.set("actionType", value);
			t.serviceUnitCountGrid.set("actionType", value);
			t.customNpiSourceGrid.set("actionType", value);
		},

		okToSave: function() {
			var t = this;
			var ok = false;

			var nameIsUnique = true;
			if (t.actionType == "Add") {
				TableManager.get("Tables.Insurance.ClaimMapping").query().forEach(function(set) {
					if (set.name == t.object.get("Name")) {
						nameIsUnique = false;
					}
				});
			}

			if (!t.formNode.validate()) {
				showMessage("Please correct any invalid fields.");
			} else if (query(".pinkBackground").length > 0) {
				showMessage("Please fill in all required fields.");
			} else if (t.actionType === "Edit" && (!t.get("modified") && !t.additionalCodingGrid.get("modified") && !t.serviceUnitCountGrid.get("modified") && !t.customNpiSourceGrid.get("modified"))) {
				showMessage("You have not made any changes.");
			} else if (!nameIsUnique) {
				showMessage('"' + t.object.get("Name") + '" is already in use. Please choose a different name.');
			} else {
				ok = true;
			}
			return ok;
		},

		save: function() {
			var t = this;
			var deferred = new Deferred();
			var promise = deferred.promise;

			if (t.okToSave()) {
				var saveData = t._getSaveData();
				var options = {
					saveData: saveData
				};

				promise = serverRequest("Tables.Insurance.ClaimMapping", "Save", options, false, 15000).then(function() {
				}, function(error) {
					showMessage(error);
					throw error;
				});
			} else {
				deferred.reject();
			}
			return promise;
		},

		_getSaveData: function() {
			var t = this;
			var saveData = t.object.getData();
			saveData.AdditionalCodes = t.additionalCodingGrid.grid.store.query();
			saveData.ServiceUnitCounts = t.serviceUnitCountGrid.grid.store.query();
			saveData.CustomNpiSources = t.customNpiSourceGrid.grid.store.query();
			return saveData;
		},
		
		resetClaimOptions: function() {
			var t = this;
			t.object.set("ProviderDetailLocation", 3);
			t.object.set("TaxonomyCodeQualifier", "");
			t.object.set("TaxonomyCode", "");
			t.object.set("TypeOfCare", "");
			t.object.set("TypeOfBillOverride", "");
			t.object.set("ProviderSignature", "");
			t.object.set("ProviderSigOnFileText", "");
			t.object.set("ProviderSigOnFileDate", "");
			t.object.set("DefaultNpiSource", "");
			t.object.set("AllowMultiplePriorAuth", true);
			t.object.set("IncludeMedicaidProviderNumber", false);
			t.object.set("TreatmentAuthorizationCode", 3);
			t.object.set("DiagnosesSource", 1);
			t.object.set("EciCodes", 1);
			t.object.set("ProceduresSource", 1);
			t.object.set("TopServiceLineWithHipps", false);
			t.object.set("TopServiceLineRevenueCode", "");
			t.object.set("TopServiceLineQuantity", "");
			t.object.set("TopServiceLineChargeAmount", "");
			t.object.set("WeeklyReportingOfGIP", true);
			t.object.set("RevenueCodeIncludeLeadingZero", true);
			t.object.set("CombineMedicationCharges", false);
			t.object.set("requireOrderingProvider", false);
			t.object.set("ServiceUnitCount", 1);
			t.object.set("ChargeAmount", 1);
			t.object.set("OtherProviderIdentifier", "");
			t.object.set("IncludeCertifyingPhysician", "");
			t.object.set("IncludeAttendingProviderTaxonomyCode", "");
			t.object.set("AttendingProviderOtherId", "");
			t.object.set("IncludeRefPhy", "");
			t.object.set("RefProviderOtherId", "");
			t.object.set("AllowDupPhy", "");
			t.object.set("CurrentIllnessDate", "");
			t.object.set("CopayAsAmountPaid", false);
			t.object.set("IncludeRoomAndBoardFacility", "");

			t.serviceUnitCountGrid.set("object", []);
			t.customNpiSourceGrid.set("object", []);
		},
		
		resetGeneralElectronicOptions: function() {
			var t = this;
			t.object.set("InterchangeSenderID", "");
			t.object.set("InterchangeReceiverQualifier", "");
			t.object.set("InterchangeReceiverID", "");
			t.object.set("RequestInterchangeAcknowledgement", false);
			t.object.set("ApplicationSenderID", "");
			t.object.set("ApplicationReceiverID", "");
		},
		
		resetAdditionalCodingOptions: function() {
			var t = this;
			t.object.set("Scenario1", false);
			t.object.set("Scenario2", false);
			t.object.set("Scenario3", false);
			t.object.set("Scenario4", false);
			t.object.set("Scenario5", false);
			t.object.set("Scenario6", false);
			t.object.set("Scenario7", false);
			t.object.set("Scenario8", false);
			t.object.set("Scenario10", false);

			t.additionalCodingGrid.set("object", []);
		},
		
		resetPatientStatementOptions: function() {
			var t = this;
			t.object.set("PayToName", "");
			t.object.set("PayToAddressStreet1", "");
			t.object.set("PayToAddressStreet2", "");
			t.object.set("PayToAddressCity", "");
			t.object.set("PayToAddressState", "");
			t.object.set("PayToAddressZip", "");
			t.object.set("PaymentPhoneNumber", "");
			t.object.set("PaymentWebsite", "");
			t.object.set("PaymentDueDate", "");
			t.object.set("PatientStatementProviderDetailLocation", 3);
			t.object.set("PatientStatementLogoPath", "");
		},
		
		resetEligibilityOptions: function() {
			var t = this;
			t.object.set("CustomInformationSourceName", "");
			t.object.set("InformationSourceIdQualifier", "");
			t.object.set("CustomInformationSourceId", "");
			t.object.set("InformationReceiverIdQualifier", "");
			t.object.set("CustomInformationReceiverID", "");
			t.object.set("InformationReceiverIndividualOrOrganizationName", "");
			t.object.set("InformationReceiverIdentificationCodeQualifier", "");
			t.object.set("InformationReceiverIdentificationCode", "");
			t.object.set("IncludeTaxonomyCode", false);
			t.object.set("CustomTaxonomyCode", "");
			t.object.set("ServiceTypeCodes", "");
		}
	});

	var additionalCodeModel = new ObjectModel({ propertyDefinitions: [
		{ name: "id", type: "text" },
		{ name: "CodeType", type: "tableItem", tableName: "FillInGroup~20620", displayName: "Type", required: true },
		{ name: "ClaimType", type: "tableItem", tableName: "FillInGroup~20621" },
		{ name: "CodeQualifier", type: "text", displayName: "Code", required: true },
		{ name: "DateFromType", type: "tableItem", tableName: "FillInGroup~20622", displayName: "Date", required: true },
		{ name: "DateFrom", type: "date", required: true },
		{ name: "DateToType", type: "tableItem", tableName: "FillInGroup~20622", displayName: "End", required: true },
		{ name: "DateTo", type: "date", required: true },
		{ name: "ValueType", type: "tableItem", tableName: "FillInGroup~20623", displayName: "Value", required: true },
		{ name: "Value", type: "text", required: true }
	]});

	var additionalCodeTemplate = {
		noForm: false,
		containers: [{
			sections: [{
				hideDivider: true,
				rows: [{
					orientation: "vertical",
					fields: [{
						name: "CodeType",
						displayName: "Code Type"
					}, {
						name: "CodeQualifier",
						displayName: "Code/Qualifier"
					}]
				}, {
					orientation: "vertical",
					fields: [{
						name: "ClaimType",
						displayName: "Claim Type"
					}, {
						name: "DateFromType",
						displayName: "Date/From"
					}, {
						name: "DateFrom",
						hideLabel: true
					}]
				}, {
					orientation: "vertical",
					fields: [{
						name: "spacer1",
						noField: true,
						hideLabel: true
					}, {
						name: "DateToType",
						displayName: "Thru"
					}, {
						name: "DateTo",
						hideLabel: true
					}]
				}, {
					orientation: "vertical",
					fields: [{
						name: "spacer2",
						noField: true,
						hideLabel: true
					}, {
						name: "ValueType",
						displayName: "Value"
					}, {
						name: "Value",
						hideLabel: true
					}]
				}]
			}]
		}]
	};

	var AdditionalCode = additionalCodeModel.getObject();
	var AdditionalCodeEditor = declare([ObjectEditor], {
		objectModel: additionalCodeModel,
		templateString: template.generate(additionalCodeTemplate),
		collection: {},

		postMixInProperties: function() {
			var t = this;
			t.inherited("postMixInProperties", arguments);

			t.getPropertyModel("DateFrom").fieldProperties.constraints = {min: MINDATE};
			t.getPropertyModel("DateTo").fieldProperties.constraints = {min: MINDATE};
		},

		buildRendering: function() {
			var t = this;
			t.inherited("buildRendering", arguments);

			domStyle.set(t.DateFromField.domNode, "width", "105px");
			domStyle.set(t.DateToField.domNode, "width", "105px");
		},

		postCreate: function() {
			var t = this;
			t.inherited(arguments);

			t.own(aspect.before(t, "_setObjectAttr", function(object) {

			}));

			t.own(aspect.after(t, "_setObjectAttr", function(object) {
				
			}, true));

			t.registerChangeLogic("CodeType", function(value) {
				t.disable("CodeQualifier", true);
				t.disable("DateFromType", true);
				t.disable("DateToType", true);
				t.disable("ValueType", true);

				if (!t.inObjectLoad) {
					t.object.set("CodeQualifier", "");
					t.object.set("DateFromType", "");
					t.object.set("DateToType", "");
					t.object.set("ValueType", "");
				}
				
				if (value == 1) { // Occurrence
					t.disable("CodeQualifier", false);
					t.disable("DateFromType", false);
				} else if (value == 2) { // Occurrence Span
					t.disable("CodeQualifier", false);
					t.disable("DateFromType", false);
					t.disable("DateToType", false);
				} else if (value == 3) { // Value
					t.disable("CodeQualifier", false);
					t.disable("ValueType", false);
				} else if (value == 4) { // Condition
					t.disable("CodeQualifier", false);
				} else if (value == 5) { // Code
					t.disable("CodeQualifier", false);
					t.disable("ValueType", false);
				}
			});

			t.registerChangeLogic("ClaimType", function(value) {

			});

			t.registerChangeLogic("CodeQualifier", function(value) {

			});

			t.registerChangeLogic("DateFromType", function(value) {
				if (value != 11) { // Custom
					t.object.set("DateFrom", "");
				}
				t.disable("DateFrom", (value != 11));
				t.require("DateFrom", (value == 11));
			});

			t.registerChangeLogic("DateFrom", function(value) {
				var constraints = t.DateToField.get("constraints");
				constraints.min = value;
				t.DateToField.set("constraints", constraints);
			});

			t.registerChangeLogic("DateToType", function(value) {
				if (value != 11) { // Custom
					t.object.set("DateTo", "");
				}
				t.disable("DateTo", (value != 11));
				t.require("DateTo", (value == 11));
			});

			t.registerChangeLogic("ValueType", function(value) {
				if (value != 6) { // Custom
					t.object.set("Value", "");
				}
				t.disable("Value", (value != 6));
				t.require("Value", (value == 6));
			});
		},

		okToSave: function() {
			var t = this;
			var ok = false;

			if (!t.formNode.validate() ) {
				showMessage("Please correct any invalid fields.");
			} else if (query(".pinkBackground", t.domNode).length > 0) {
				showMessage("Please fill in all required fields.");
			} else if (t.actionType === "Edit" && !t.get("modified")) {
				showMessage("You have not made any changes.");
			} else {
				ok = true;
			}
			return ok;
		}
	});

	var ItemGrid = declare([_WidgetBase, _TemplatedMixin, _WidgetsInTemplateMixin], {
		title: "Edit Item",
		gridColumns: null,
		ModelType: null,
		EditorType: null,
		ObjectType:null,
		
		grid: {},
		store: new Memory(),
		buttons: [],
		modified: false,

		templateString: '<div>'+
							'<div class="dcs-subsmall" data-dojo-attach-point="titleNode"></div>'+
							'<div data-dojo-attach-point="actions"></div>'+
							'<div data-dojo-attach-point="gridContainer"></div>'+
						'</div>',

		postMixInProperties: function() {
			var t = this;
			t.inherited("postMixInProperties", arguments);
		},

		buildRendering: function() {
			var t = this;
			t.inherited("buildRendering", arguments);

			t.createGrid();
			t.actionButtons();

			domStyle.set(t.grid.domNode, "width", "750px");
			domStyle.set(t.grid.domNode, "height", "150px");

			t.gridContainer.appendChild(t.grid.domNode);
			t.grid.startup();
		},

		postCreate: function() {
			var t = this;
			t.inherited("postCreate", arguments);

			t.grid.connect(t.grid, "onCellDblClick", lang.hitch(t, t.actionEdit));

			t.own(aspect.after(t, "_setObjectAttr", function(object) {
				t.grid.setStore(new Memory({ data: object }));
				t.refreshGridBody();
			}, true));

			t.refreshGridBody();
			t.grid.resize();
		},

		getLayout: function() {
			var t = this;
			var properties = t.gridColumns;
			return t.ModelType.getGridLayout(properties);
		},
		
		createGrid: function() {
			var t = this;

			t.grid = new Grid({
				cacheClass: Cache,
				store: t.store,
				structure: t.getLayout(),
				selectRowMultiple: false,
				selectRowTriggerOnCell: true,
				modules: [
					VScroller,
					SelectRow,
					RowHeader,
					IndirectSelect,
					SingleSort
				]
			});

			t.own(aspect.after(t.grid, "_setDisabledAttr", lang.hitch(t, function(disable) {
				if (disable) {
					for (var i = 0; i < t.grid.store.data.length; i++) {
						t.grid.select.row.unselectable[t.grid.store.data[i].id] = 1;
					}
				} else {
					t.grid.select.row.unselectable = {};
				}
				t.grid.body.refresh();
			}), true));
		},

		refreshGridBody: function() {
			var t = this;
			t.grid.select.row.clear();
			t.grid.vScroller.scrollToRow(0,1);
			t.grid.model.clearCache();
			t.grid.body.refresh();
		},

		_setActionTypeAttr: function(value) {
			/*
				value = "Edit", "Add", or "View"
			*/
			
			var t = this;
			t.inherited(arguments);
			t.grid.set("disabled",  (value == "View"));

			arrayUtils.forEach(t.buttons,function(button) {
				button.set("disabled", (value == "View"));
			});
		},

		openEditor: function(object, actionType) {
			var t = this;
			var editor = new t.EditorType();
			editor.set("object", object);
			editor.set("actionType", actionType);
			editor.set("collection", t.grid.store.data);
			//TODO editor.set("claimMappingEditor", t);

			var messageDialog = new Dialog({
				title: t.title,
				content: editor,
				style: "width: 755px;",
				onCancel: function() {
					this.destroyRecursive();
				}
			});

			var actionBar = domConstruct.toDom("<div class='dijitDialogPaneActionBar'></div>");
			domConstruct.place(actionBar,messageDialog.containerNode);
			
			var closeDialog = function() {
				messageDialog.hide();
				messageDialog.destroyRecursive();
			};

			new Button({
				label: "Save",
				onClick: function() {
					if (editor.okToSave()) {
						var data = editor.get("object").getData();
						if (typeof data.id == "undefined") {
							var maxId = 0;
							arrayUtils.forEach(t.grid.store.data, function(code) {
								maxId = Math.max(code.id, maxId);
							});
							data.id = maxId + 1;
						}
						t.grid.store.put(data);
						t.refreshGridBody();
						t.grid.select.row.selectById(data.id);
						closeDialog();
						t.modified = true;
					}
				}
			}).placeAt(actionBar);

			new Button({
				label: "Cancel",
				onClick: closeDialog
			}).placeAt(actionBar);
			messageDialog.show();
		},

		actionButtons: function() {
			var t = this;

			t.buttons.push(new Button({
				label: "Add",
				disabled: t.readOnly,
				onClick: function() { t.actionAdd(); }
			}).placeAt(t.actions));

			t.buttons.push(new Button({
				label: "Edit",
				disabled: t.readOnly,
				onClick: function() { t.actionEdit(); }
			}).placeAt(t.actions));

			t.buttons.push(new Button({
				disabled: t.readOnly,
				label: "Remove",
				onClick: function() { t.actionRemove(); }
			}).placeAt(t.actions));
		},

		actionAdd: function() {
			var t = this;
			t.openEditor(new t.ObjectType(), "Add");
		},

		actionEdit: function() {
			var t = this;
			var selection = t.grid.select.row.getSelected();
			if (selection.length > 0) {
				t.openEditor(new t.ObjectType(t.grid.store.get(selection)), "Edit");
			}
		},

		actionRemove: function() {
			var t = this;
			var selection = t.grid.select.row.getSelected();
			if (selection.length > 0) {
				t.grid.store.remove(selection);
				t.refreshGridBody();
				t.modified = true;
			}
		}
	});

	var AdditionalCodeGrid = declare(ItemGrid,{
		title: "Additional Code",
		gridColumns: ["CodeType", "CodeQualifier", "DateFromType", "DateToType", "ValueType"],
		ModelType: additionalCodeModel,
		EditorType: AdditionalCodeEditor,
		ObjectType: AdditionalCode,
		
		getLayout: function() {
			var t = this;
			var layout = t.inherited(arguments);

			var valueFormatter = function(object) {
				var valueName = this.id.substr(0, this.id.length - 4);
				var propertyModel = t.ModelType.getPropertyModel(this.id);
				var tableFormatter = lang.hitch(propertyModel, propertyModel.formatter);
				var valueFormatter = t.ModelType.getPropertyModel(valueName).formatter;
				var formattedValue;

				if (tableFormatter(object[this.id]) == "Custom") {
					formattedValue = valueFormatter(object[valueName]);
				} else {
					formattedValue = tableFormatter(object[this.id]);
				}
				
				return formattedValue;
			};

			layout[2].formatter = valueFormatter;
			layout[3].formatter = valueFormatter;
			layout[4].formatter = valueFormatter;

			return layout;
		}
	});
	
	var serviceUnitCountModel = new ObjectModel({ propertyDefinitions: [
		{ name: "id", type: "text" },
		{ name: "Default", type: "boolean", displayName: "Default" },
		{ name: "Discipline", type: "tableItem", tableName: "Tables.Discipline", displayName: "Discipline", required: true },
		{ name: "ServiceUnitCount", type: "tableItem", tableName: "FillInGroup~20618", displayName: "Service Unit Count", required: true },
		{ name: "ServiceUnitsPerCharge", type: "text", displayName: "Units Per Charge" }
	]});

	var serviceUnitCountTemplate = {
		noForm: false,
		containers: [{
			sections: [{
				hideDivider: true,
				rows: [{
					field: {
						name: "Default",
						displayName: "Default",
						fieldSize: "small",
						labelSize: "large"
					}
				}, { 
					field: {
						name: "Discipline",
						displayName: "Discipline",
						fieldSize: "large",
						labelSize: "large"
					}
				}, { 
					fields: [{
						name: "ServiceUnitCount",
						displayName: "Service Unit Count",
						fieldSize: "medium-small",
						labelSize: "large"
					},{
						name: "ServiceUnitsPerCharge",
						displayName: "Units Per Charge",
						fieldSize: "small",
						labelSize: "large"
					}]
				}]
			}]
		}]
	};

	var ServiceUnitCount = serviceUnitCountModel.getObject();
	var ServiceUnitCountEditor = declare([ObjectEditor], {
		objectModel: serviceUnitCountModel,
		templateString: template.generate(serviceUnitCountTemplate),
		collection: {},

		postMixInProperties: function() {
			var t = this;
			t.inherited("postMixInProperties", arguments);

			t.getPropertyModel("ServiceUnitsPerCharge").getField = function(node) {
				return new NumberTextBox(lang.mixin(this.getFieldProperties(), {constraints: {places: "0", min: 0, max: 999}}), node);
			};
			t.getPropertyModel("ServiceUnitsPerCharge").getInternal = function(value) {
				return (value ? value : (value === 0 ? value : ""));
			};
		},

		buildRendering: function() {
			var t = this;
			t.inherited("buildRendering", arguments);
			
			domStyle.set(t.ServiceUnitsPerChargeField.domNode, "width", "60px");
		},

		postCreate: function() {
			var t = this;
			t.inherited(arguments);

			t.registerChangeLogic("Default", function(value) {
				t.disable("Discipline", (value == 1));
				if (!t.inObjectLoad) {
					t.object.set("Discipline","");
				}
			});
			
			t.registerChangeLogic("ServiceUnitCount", function(value) {
				domStyle.set(t.ServiceUnitsPerChargeContainer, "display", value == 1 ? "" : "none");
				t.require("ServiceUnitsPerCharge", value == 1);
				t.object.set("ServiceUnitsPerCharge", value == 1 ? 1 : "");
			});
		},

		okToSave: function() {
			var t = this;
			var ok = false;

			if (!t.formNode.validate() ) {
				showMessage("Please correct any invalid fields.");
			} else if (query(".pinkBackground", t.domNode).length > 0) {
				showMessage("Please fill in all required fields.");
			} else if (t.actionType === "Edit" && !t.get("modified")) {
				showMessage("You have not made any changes.");
			} else {
				ok = true;
			}
			return ok;
		},

		_getObjectAttr: function() {
			var t = this;
			var object = t.inherited(arguments);
			if (typeof object.Default === "undefined") {
				object.Default = 0;
			}
			return object;
		},

		_setObjectAttr: function(object) {
			var t = this;
			t.inherited(arguments);

			// Exclude "Custom"
			var inclusions = [];
			var propertyModel = t.getPropertyModel("ServiceUnitCount");
			arrayUtils.forEach(propertyModel.getOptionStore().data,function(item) {
				if (item.name !== "Custom") {
					inclusions.push(item.id);
				}
			});
			t.populateSelect(t.ServiceUnitCountField,propertyModel.getOptionStore(),(object ? object.ServiceUnitCount : null),true,inclusions);
		},

		_setCollectionAttr: function(value) {
			var t = this;

			// Only allow a single default selection
			t.disable("Default",arrayUtils.some(value,function(item) { return item.Default; }));
			
			// Exclude disciplines that have already been added
			var inclusions = [];
			var propertyModel = t.getPropertyModel("Discipline");
			arrayUtils.forEach(propertyModel.getOptionStore().data,function(discipline) {
				if (!arrayUtils.some(value,function(item) { return (!t.object || t.object.id !== item.id) && item.Discipline == discipline.id; })) {
					inclusions.push(discipline.id);
				}
			});
			t.populateSelect(t.DisciplineField,propertyModel.getOptionStore(),(t.object ? t.object.Discipline : null),true,inclusions);
		}
	});
	
	var ServiceUnitCountGrid = declare(ItemGrid,{
		title: "Service Unit Count",
		gridColumns: ["Default","Discipline","ServiceUnitCount","ServiceUnitsPerCharge"],
		ModelType: serviceUnitCountModel,
		EditorType: ServiceUnitCountEditor,
		ObjectType: ServiceUnitCount
	});
	
	var npiSourceModel = new ObjectModel({ propertyDefinitions: [
		{ name: "id", type: "text" },
		{ name: "Discipline", type: "tableItem", tableName: "CustomNpiDisciplineList", displayName: "Discipline", required: true },
		{ name: "NpiSource", type: "tableItem", tableName: "FillInGroup~20653", displayName: "NPI Source", required: true }
	]});

	var npiSourceTemplate = {
		noForm: false,
		containers: [{
			sections: [{
				hideDivider: true,
				rows: [{ 
					field: {
						name: "Discipline",
						displayName: "Discipline",
						fieldSize: "large",
						labelSize: "large"
					}
				}, { 
					field: {
						name: "NpiSource",
						displayName: "NPI Source",
						fieldSize: "medium-small",
						labelSize: "large"
					}
				}]
			}]
		}]
	};

	var NpiSource = npiSourceModel.getObject();
	var NpiSourceEditor = declare([ObjectEditor], {
		objectModel: npiSourceModel,
		templateString: template.generate(npiSourceTemplate),
		collection: {},

		postMixInProperties: function() {
			var t = this;
			t.inherited("postMixInProperties", arguments);
		},

		buildRendering: function() {
			var t = this;
			t.inherited("buildRendering", arguments);
		},

		postCreate: function() {
			var t = this;
			t.inherited(arguments);

			t.registerChangeLogic("Discipline", function(value) {
				// Exclude "Employee" when Level of Care or Room & Board are selected
				var includeEmployee = true;
				if ((value == "extraLevelOfCare") || (value == "extraRoomAndBoard")) {
					includeEmployee = false;
				}
				var inclusions = [];
				var propertyModel = t.getPropertyModel("NpiSource");
				arrayUtils.forEach(propertyModel.getOptionStore().data,function(item) {
					if ((item.name !== "Employee") || (includeEmployee)) {
						inclusions.push(item.id);
					}
				});
				t.populateSelect(t.NpiSourceField,propertyModel.getOptionStore(),null,true,inclusions);
			});
		},

		okToSave: function() {
			var t = this;
			var ok = false;

			if (!t.formNode.validate() ) {
				showMessage("Please correct any invalid fields.");
			} else if (query(".pinkBackground", t.domNode).length > 0) {
				showMessage("Please fill in all required fields.");
			} else if (t.actionType === "Edit" && !t.get("modified")) {
				showMessage("You have not made any changes.");
			} else {
				ok = true;
			}
			return ok;
		},

		_setCollectionAttr: function(value) {
			var t = this;
			
			// Exclude disciplines that have already been added
			var inclusions = [];
			var propertyModel = t.getPropertyModel("Discipline");
			arrayUtils.forEach(propertyModel.getOptionStore().data,function(discipline) {
				if (!arrayUtils.some(value,function(item) { return (!t.object || t.object.id !== item.id) && item.Discipline == discipline.id; })) {
					inclusions.push(discipline.id);
				}
			});
			t.populateSelect(t.DisciplineField,propertyModel.getOptionStore(),(t.object ? t.object.Discipline : null),true,inclusions);
		}
	});

	var CustomNpiSourceGrid = declare(ItemGrid,{
		title: "Custom NPI Source",
		gridColumns: ["Discipline","NpiSource"],
		ModelType: npiSourceModel,
		EditorType: NpiSourceEditor,
		ObjectType: NpiSource
	});

	return ClaimMappingEditor;
});